#!/bin/sh -x

export FS_SOURCE_PATH="$1"
export DEST_DRIVE="/dev/mmcblk0"
export FS_SOURCE="$2"

export BOOT_DEV=""
export BOOT_FS="ext2"
export BOOT_MOUNT="/mnt/root/boot"

export ROOT_DEV="${DEST_DRIVE}p2"
export ROOT_FS="ext3"
export ROOT_MOUNT="/mnt/root"

export DATA_SIZE="3000M"
export DATA_DEV="${DEST_DRIVE}p3"
export DATA_FS="ext3"
export DATA_MOUNT="/mnt/root/data"

export SHADOW_DEV="${DEST_DRIVE}p4"
export SHADOW_FS="ext3"
export SHADOW_MOUNT="/mnt/root/var/shadow"

export LOG_PATH="${FS_SOURCE_PATH}/log"
export LOG_CONSOLE=${LOG_PATH}/install.log

export PATH=/sbin:/bin:/usr/sbin:/usr/bin

do_create_partition_table() {
fdisk -u "$DEST_DRIVE" << EOF
d
1
d
2
d
3
d
n
p
1
16
3967
n
p
2

+690M
n
p
3

+$(echo "$DATA_SIZE")
n
p
4


w
EOF
cat /proc/partitions
}

check_datafs_changed() {
    test "$(partinfo "$DATA_DEV" | awk -F '[:,=]' '{print $3}')" != 1351625
}

do_restore_logo() {
    # Install logo.bmp if provided
    if [ ! -z "${logo_file}" ]; then
      mv "${logo_file}" "${ROOT_MOUNT}/boot/logo.bmp"
    fi
}

do_backup_logo() {
    # If logo.bmp file is not specified in Console.ini, delete it (don't install)
    if grep -q "SplashBoot=true" ${ROOT_MOUNT}/data/x30/Console.ini ; then
      export logo_file="${FS_SOURCE_PATH}/logo_${time}.bmp"
      cp "${ROOT_MOUNT}/boot/logo.bmp" "${logo_file}"
    fi
}

do_set_boot_mode() {
    # Set boot mode into sunrise
    /usr/bin/SetBootMode 0

    # Disable kernel serial output by setting console mode to 1
    /usr/bin/SetConsoleMode 1
}

do_check_disk_size() {
    # Check if disk size is over 5.6GB (11000000 * 512)
    if [ "$(cat /sys/class/block/mmcblk0/size)" -gt 11000000 ]; then
      export DATA_SIZE="4000M"
    fi
}
REFORMAT_DATA=0
REFORMAT_SHADOW=0

get_machine() {
    basename "$1" | awk -F- '{
      if ($1 == "egspro") printf "%s-%s", $1, $2
      else print $1
    }'
}

MACHINE=`get_machine "${FS_SOURCE}"`

partinfo() {
    sfdisk -d "$DEST_DRIVE" 2>/dev/null | grep "^$1" | sed 's/ //g'
}

fstype() {
    (blkid -p -n ext3 -o export $1 2>/dev/null; echo 'echo $TYPE') | sh
}

# umount all the partitions
do_unmount_target() {
    umount $BOOT_DEV
    umount $SHADOW_DEV
    umount $DATA_DEV
    umount $ROOT_DEV
}

# mount target filesystems
do_mount_target() {
    # mount root first
    mkdir -p $ROOT_MOUNT
    mount $ROOT_DEV $ROOT_MOUNT -t $ROOT_FS

    # make the directories under root
    mkdir -p $BOOT_MOUNT
    mkdir -p $SHADOW_MOUNT
    mkdir -p $DATA_MOUNT

    # mount the others
    test ! ${BOOT_DEV} || mount $BOOT_DEV $BOOT_MOUNT -t $BOOT_FS
    test ! ${SHADOW_DEV} || mount $SHADOW_DEV $SHADOW_MOUNT -t $SHADOW_FS
    mount $DATA_DEV $DATA_MOUNT
}

verify_checksums() {
    (cd "${FS_SOURCE_PATH}" && sha1sum -c "${MACHINE}-checksums.txt")
}

do_console_details() {
    echo "Console details"
    echo "Brand $(brand)"
    echo "Machine $(brand -m)"
    echo "Serial Number $(brand -e)"
    echo "UUID $(brand -q)"
    echo "Install version $(brand -V)"
    echo "Installing Brand ${INSTALL_BRAND}"
}

get_brand() {
    echo "$(brand -c ${FS_SOURCE_PATH}/Console.ini)"
}

INSTALL_BRAND=$(get_brand)

do_data_setup() {
  cd "${ROOT_MOUNT}/data"  
  if [ -d x30 ]; then
    # Clean out any old External Application settings
    rm -r x30/RemoteView

    # Clean out any developer settings
    rm x30/Software.ini
  fi
  cd -
}

# create filesystems on devices
do_create_file_systems() {
    test "REFORMAT_SHADOW" = 0 || mkfs.$SHADOW_FS -F $SHADOW_DEV
    test ! ${BOOT_DEV} || mkfs.$BOOT_FS -F $BOOT_DEV

    mkfs.$ROOT_FS -F $ROOT_DEV
    test "$REFORMAT_DATA" = 0 || mkfs.$DATA_FS -F $DATA_DEV
}

# extract the image
do_extract_fs_image() {
    cd $ROOT_MOUNT
    tar -pxzf "${FS_SOURCE_PATH}/${FS_SOURCE}" --warning=no-timestamp
    cp "${FS_SOURCE_PATH}/pkginfo.json" "${ROOT_MOUNT}/usr/share/horizon/pkginfo.json"
    cp "${FS_SOURCE_PATH}/pkginfo.sign" "${ROOT_MOUNT}/usr/share/horizon/pkginfo.sign"
    cd -
}

do_happ_perms() {
    chroot $ROOT_MOUNT chown -R happ:happ /data/happ/apps /data/happ/data /data/happ/repo
}

# Ensure all written to disk for reboot
do_restore_backup() {
    if [ -x "${backup_file}" ]; then
      cd "${DATA_MOUNT}"
      tar --use-compress-program=lzop -pxf "${backup_file}"
      cd -
    fi

    cd "${DATA_MOUNT}"
    sed -i "/\[Software\]/d" x30/Console.ini
    sed -i "/Developer=/d" x30/Console.ini
    if [ -x "${FS_SOURCE_PATH}/Console.ini" ]; then
      sed -i "/\[Console\]/d" x30/Console.ini
      sed -i "/Brand=/d" x30/Console.ini
      cat "${FS_SOURCE_PATH}/Console.ini" >> x30/Console.ini
    fi
    cd -

    sync
}

# Restore production settings
do_restore_production() {
    # Write Console.ini
    cd "${DATA_MOUNT}"
    if [ -x "${FS_SOURCE_PATH}/Console.ini" ]; then
      sed -i "/\[Console\]/d" x30/Console.ini
      sed -i "/Brand=/d" x30/Console.ini
      cat "${FS_SOURCE_PATH}/Console.ini" >> x30/Console.ini
    fi
    cd -

    # Write factory settings
    settings_folder="$(brand -x -c ${DATA_MOUNT}/x30/Console.ini)"
    cp -r ${ROOT_MOUNT}/usr/share/horizon/${settings_folder}/* ${DATA_MOUNT}/x30/.
}

do_install_brand() {
    # Install hostname from hostnames file if provided
    MAC=$( ifconfig eth0 2>/dev/null | grep HWaddr | sed 's/.*HWaddr\ *\([a-zA-Z0-9:]*\).*/\1/' 2>/dev/null)
    if [ ! -z "${MAC}" ]; then
      USB_HOSTNAME=$( grep -i $MAC ${FS_SOURCE_PATH}/hostnames 2>/dev/null | cut -d' ' -f2)
    fi

    if [ ! -z "${USB_HOSTNAME}" ]; then
      # Found MAC address in hostnames file
      echo "${USB_HOSTNAME}" > "${ROOT_MOUNT}/etc/hostname"
      rm "${hostname_file}"
    elif [ -f "${hostname_file}" ]; then 
      # Did not find MAC address in hostnames file update file but have backup hostname file
      if [ -f "${FS_SOURCE_PATH}/hostnames" ]; then
        echo "${MAC} $(cat ${hostname_file})" >> ${FS_SOURCE_PATH}/hostnames
      fi
      mv "${hostname_file}" "${ROOT_MOUNT}/etc/hostname"
    fi

    # if user hostname file is provided, use that as default regardless of what is in hostnames	file
    if [ -f "${FS_SOURCE_PATH}/hostname" ]; then
      cp "${FS_SOURCE_PATH}/hostname" "${ROOT_MOUNT}/etc/hostname"
    fi

    # Install angstrom_feeds if provided
    if [ -f "${FS_SOURCE_PATH}/angstrom_feeds.conf" ]; then
      cp "${FS_SOURCE_PATH}/angstrom_feeds.conf" "${ROOT_MOUNT}/etc/opkg/angstrom_feeds.conf"
    fi

    # Install wired.config if provided
    if [ -f "${FS_SOURCE_PATH}/wired.config" ]; then
      cp "${FS_SOURCE_PATH}/wired.config" "${ROOT_MOUNT}/var/lib/connman/wired.config"
    fi
}

do_install_certificates() {
  # File Server or Printer
  if [ -f "${FS_SOURCE_PATH}/fs_server-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/fs_server-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/fs_server-cert.bin"
  fi

  # ISO Client
  if [ -f "${FS_SOURCE_PATH}/iso_client-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/iso_client-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/iso_client-cert.bin"
  fi

  # Task Controller
  if [ -f "${FS_SOURCE_PATH}/tc_server-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/tc_server-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/tc_server-cert.bin"
  fi

  # Virtual Terminal
  if [ -f "${FS_SOURCE_PATH}/vt_server-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/vt_server-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/vt_server-cert.bin"
  fi

  # ID Server
  if [ -f "${FS_SOURCE_PATH}/id_server-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/id_server-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/id_server-cert.bin"
  fi

  # Proprietary MCF
  if [ -f "${FS_SOURCE_PATH}/proprietary_mcf-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/proprietary_mcf-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/proprietary_mcf-cert.bin"
  fi

  # Tractor ECU
  if [ -f "${FS_SOURCE_PATH}/tecu_server-cert.bin" ]; then
    mkdir -p "${ROOT_MOUNT}/usr/share/isoman/certificates"
    cp "${FS_SOURCE_PATH}/tecu_server-cert.bin" "${ROOT_MOUNT}/usr/share/isoman/certificates/tecu_server-cert.bin"
  fi
}

do_install_extras() {
    local list=""
    local cmd='cat > /tmp/$TAR_FILENAME && opkg install --nodeps -o ${ROOT_MOUNT} /tmp/$TAR_FILENAME ; rm -f /tmp/$TAR_FILENAME'

    # Pipe the extras into opkg install
    list="$(tar --wildcards -xf ${FS_SOURCE_PATH}/${MACHINE}-extras.img 'extras.*.txt' --to-command cat | sort | uniq)"
    tar -xf ${FS_SOURCE_PATH}/${MACHINE}-extras.img ${list} --to-command "${cmd}"

    # Finalize the install
    chroot ${ROOT_MOUNT} opkg configure
}

do_install_hpkgs() {
    export HAPPDIR=${DATA_MOUNT}/happ
    ls ${FS_SOURCE_PATH}/hpkgs/*.hpkg 2>/dev/null | while read pkg
    do
        uid=$(${ROOT_MOUNT}/usr/bin/hpkg info -p $pkg | awk '/^uid:/{print $2}')
        if ! ls ${HAPPDIR}/apps/${uid}/* >/dev/null 2>&1
        then
                ${ROOT_MOUNT}/usr/bin/hpkg install $pkg
        fi
    done
}

do_reboot() {

    do_unmount_target
    umount "${FS_SOURCE_PATH}"

    # poweroff for production build
    # poweroff -f && while true; do sleep 1000; done

    # Remove Topcon signature
    usb_drive_mbr="$(echo ${FS_SOURCE_PATH} | sed -n -r 's/\/media(\/sd[a-z]{1})[0-9]+\//\/dev\1/p')"
    echo -e -n "\000\000\000\000\000\000" | dd of="${usb_drive_mbr}" seek=440 bs=1 count=6
    sync

    # reboot for normal upgrade
    reboot -f && while true; do sleep 1000; done
}

do_backup() {
    # Generate unique file name for the backup
    time=$(date +%s)
    export backup_file="${FS_SOURCE_PATH}/backup_${time}.tar.lzo"

    # Backup user data
    cd "${ROOT_MOUNT}/data"
    if [ -d x30 ]; then
        # Build the exclude list file
        echo "x30/.FactoryVehicles*"        > /tmp/exclude_list
        echo "x30/.FactoryImplements*"     >> /tmp/exclude_list
        echo "x30/.FactoryProducts*"       >> /tmp/exclude_list
        echo "x30/.FactoryNozzles*"        >> /tmp/exclude_list
        echo "x30/.usbkey-full-on-upgrade" >> /tmp/exclude_list
        echo "x30/FactoryVehicles*"        >> /tmp/exclude_list
        echo "x30/FactoryImplements*"      >> /tmp/exclude_list
        echo "x30/FactoryProducts*"        >> /tmp/exclude_list
        echo "x30/FactoryNozzles*"         >> /tmp/exclude_list
        echo "x30/oem_profile.conf"        >> /tmp/exclude_list
        echo "x30/UserLevelsFactory.ini"   >> /tmp/exclude_list
        echo "x30/cache"                   >> /tmp/exclude_list
        echo "x30/Trolltech.conf"          >> /tmp/exclude_list
        echo "x30/CrashReports"            >> /tmp/exclude_list
        echo "x30/Diagnostics"             >> /tmp/exclude_list
        echo "x30/Sounds"                  >> /tmp/exclude_list
        echo "x30/RemoteView"              >> /tmp/exclude_list
        echo "x30/backup_*.tar.gz"         >> /tmp/exclude_list
        echo "x30/backup_*.tar.lzo"        >> /tmp/exclude_list
        echo "x30/cache"                   >> /tmp/exclude_list
        echo "x30/agi1-fw"                 >> /tmp/exclude_list
        echo "x30/agi2-fw"                 >> /tmp/exclude_list
        echo "x30/agi3-fw"                 >> /tmp/exclude_list
        echo "x30/agi4-fw"                 >> /tmp/exclude_list
        echo "x30/sgr1-fw"                 >> /tmp/exclude_list
        echo "x30/asc10-fw"                >> /tmp/exclude_list
        echo "x30/Software.ini"            >> /tmp/exclude_list
        echo "x30/*.csexe"                 >> /tmp/exclude_list
        echo "x30/*.disabled"              >> /tmp/exclude_list
        echo "x30/*.enabled"               >> /tmp/exclude_list
        echo "x30/smb.inc"                 >> /tmp/exclude_list

        dirs=""
        dirs="$dirs $(test -d x30 && echo x30)"
        dirs="$dirs $(test -d happ/data && echo happ/data)"

        tar -X /tmp/exclude_list --use-compress-program=lzop -pcf "${backup_file}" ${dirs}
        rm -f /tmp/exclude_list

        if [ "$?" != "0" ]; then
          rm "${backup_file}"
          touch "x30/.usbkey-full-on-upgrade"
          do_reboot
        fi

        sync
    fi
    cd -
}

do_etc_backup() {
    time=$(date +%s)
    export etc_backup_file="${FS_SOURCE_PATH}/etc_backup_${time}.tar.gz"
    export hostname_file="/tmp/hostname_${time}"

    # Store hostname file
    if [ -f "${FS_SOURCE_PATH}/hostname" ]; then
      cp "${FS_SOURCE_PATH}/hostname" "${hostname_file}"
    fi

    # Backup existing hostname, angstrom_feeds.conf and wired.config
    etc_files=""
    etc_files="${etc_files} /etc/hostname"
    etc_files="${etc_files} /etc/opkg/angstrom_feeds.conf"
    etc_files="${etc_files} /var/lib/connman/wired.config"
    tar -C "${ROOT_MOUNT}" -pzcf "${etc_backup_file}" ${etc_files}

    do_backup_logo
}

do_etc_restore() {
    etc_auto_restore_hint="${FS_SOURCE_PATH}/auto_etc_restore"

    if [ -f "${etc_auto_restore_hint}" -a -f "${etc_backup_file}" ]; then
        tar -C "${ROOT_MOUNT}" -pzxf "${etc_backup_file}"
    fi
}

do_delete_pclient_project() {
    cd "${ROOT_MOUNT}/data"
    if [ -d pclient ]; then
        # don't want to keep PClient project
        rm -rf pclient/projekte/default_prj/
        sync
    fi
    cd -
}

progressbar() {
    case "$INSTALL_BRAND" in
      "McHale")
        /usr/bin/upgrade-splash-mchale $1 2>&1 >/dev/null
        ;;
      *)
        /usr/bin/upgrade-splash $1 2>&1 >/dev/null
        ;;
    esac
}

do_full_install() {
    progressbar 0
    if ! verify_checksums
    then
        return 1
    fi

    progressbar 10
    do_console_details

    if test -n "$(fstype "$DATA_DEV")"; then
        fsck.ext3 -p "$DATA_DEV" || :

        if ! fsck.ext3 -n "$DATA_DEV" || check_datafs_changed; then
            REFORMAT_DATA=1
        fi

        if test "$REFORMAT_DATA" = 1; then
            do_mount_target
            do_etc_backup
            do_backup
            do_unmount_target
        else
            do_mount_target
            do_delete_pclient_project
            do_etc_backup
            do_unmount_target
        fi
    else
        # Check if perhaps the file system isn't ext3
        if fsck.ext3 -n "$DATA_DEV"; then
            do_mount_target
            do_etc_backup
            do_backup
            do_unmount_target
        fi

        REFORMAT_DATA=1
    fi

    if test -n "$(fstype "$SHADOW_DEV")"; then
        fsck.ext3 -p "$SHADOW_DEV" || :

        if ! fsck.ext3 -n "$SHADOW_DEV"; then
            REFORMAT_SHADOW=1
        fi
    else
        test ! ${SHADOW_DEV} || REFORMAT_SHADOW=1
    fi

    progressbar 20
    do_create_partition_table

    progressbar 30
    do_create_file_systems
    do_mount_target
    do_data_setup

    progressbar 40
    do_extract_fs_image
    do_etc_restore

    progressbar 50
    do_install_brand

    progressbar 60
    do_install_extras
    do_restore_logo
    do_install_certificates

    progressbar 70
    do_install_hpkgs

    progressbar 80
    do_happ_perms

    progressbar 90
    do_restore_backup

    progressbar 100
}

if [ "$3" == "run" ]; then
    do_check_disk_size
    if do_full_install
    then
        do_set_boot_mode
    fi
    do_reboot
else
    mount -t tmpfs tmpfs /tmp
    cp -f "$0" /tmp/install
    chmod +x /tmp/install
    mkdir -p "${LOG_PATH}"
    exec /tmp/install "$1" "$2" run 2>&1 | tee "$LOG_CONSOLE"
fi
