#!/bin/sh -x

export FS_SOURCE_PATH="$1"
export DEST_DRIVE="/dev/mmcblk0"
export FS_SOURCE="$2"

export BOOT_DEV=""
export BOOT_FS="ext2"
export BOOT_MOUNT="/mnt/root/boot"

export ROOT_DEV="${DEST_DRIVE}p2"
export ROOT_FS="ext3"
export ROOT_MOUNT="/mnt/root"

export DATA_SIZE="4000M"
export DATA_DEV="${DEST_DRIVE}p3"
export DATA_FS="ext3"
export DATA_MOUNT="/mnt/root/data"

export SHADOW_DEV="${DEST_DRIVE}p4"
export SHADOW_FS="ext3"
export SHADOW_MOUNT="/mnt/root/var/shadow"

export LOG_PATH="${FS_SOURCE_PATH}/log"
export LOG_CONSOLE=${LOG_PATH}/install.log

export PATH=/sbin:/bin:/usr/sbin:/usr/bin

do_create_partition_table() {
fdisk -u "$DEST_DRIVE" << EOF
d
1
d
2
d
3
d
n
p
1
16
3967
n
p
2

+640M
n
p
3

+$(echo "$DATA_SIZE")
n
p
4


w
EOF
cat /proc/partitions
}

check_datafs_changed() {
    test "$(partinfo "$DATA_DEV" | awk -F '[:,=]' '{print $3}')" != 1253969
}

do_restore_logo() {
    # Install logo.bmp if provided
    if [ ! -z "${logo_file}" ]; then
      mv "${logo_file}" "${ROOT_MOUNT}/boot/logo.bmp"
    fi
}

do_backup_logo() {
    # If logo.bmp file is not specified in Console.ini, delete it (don't install)
    if grep -q "SplashBoot=true" ${ROOT_MOUNT}/data/x30/Console.ini ; then
      export logo_file="${FS_SOURCE_PATH}/logo_${time}.bmp"
      cp "${ROOT_MOUNT}/boot/logo.bmp" "${logo_file}"
    fi
}

do_set_boot_mode() {
    # Set boot mode into sunrise
    /usr/bin/SetBootMode 0

    # Disable kernel serial output by setting console mode to 1
    /usr/bin/SetConsoleMode 1
}

do_check_disk_size() {
    test ! "$(fdisk -l /dev/mmcblk0 | grep "3909 MB")" || export DATA_SIZE="3000M"
}
REFORMAT_DATA=0
REFORMAT_SHADOW=0

partinfo() {
    sfdisk -d "$DEST_DRIVE" 2>/dev/null | grep "^$1" | sed 's/ //g'
}

fstype() {
    (blkid -p -n ext3 -o export $1 2>/dev/null; echo 'echo $TYPE') | sh
}

# umount all the partitions
do_unmount_target() {
    umount $BOOT_DEV
    umount $SHADOW_DEV
    umount $DATA_DEV
    umount $ROOT_DEV
}

# mount target filesystems
do_mount_target() {
    # mount root first
    mkdir -p $ROOT_MOUNT
    mount $ROOT_DEV $ROOT_MOUNT -t $ROOT_FS

    # make the directories under root
    mkdir -p $BOOT_MOUNT
    mkdir -p $SHADOW_MOUNT
    mkdir -p $DATA_MOUNT

    # mount the others
    test ! ${BOOT_DEV} || mount $BOOT_DEV $BOOT_MOUNT -t $BOOT_FS
    test ! ${SHADOW_DEV} || mount $SHADOW_DEV $SHADOW_MOUNT -t $SHADOW_FS
    mount $DATA_DEV $DATA_MOUNT
}

do_console_details() {
    echo "Console details"
    echo "Brand $(brand)"
    echo "Machine $(brand -m)"
    echo "Serial Number $(brand -e)"
}

# create filesystems on devices
do_create_file_systems() {
    test "REFORMAT_SHADOW" = 0 || mkfs.$SHADOW_FS -F $SHADOW_DEV
    test ! ${BOOT_DEV} || mkfs.$BOOT_FS -F $BOOT_DEV

    mkfs.$ROOT_FS -F $ROOT_DEV
    test "$REFORMAT_DATA" = 0 || mkfs.$DATA_FS -F $DATA_DEV
}

# extract the image
do_extract_fs_image() {
    cd $ROOT_MOUNT
    tar -pxzf "${FS_SOURCE_PATH}/${FS_SOURCE}" --warning=no-timestamp
    cp "${FS_SOURCE_PATH}/pkginfo.json" "${ROOT_MOUNT}/usr/share/horizon/pkginfo.json"
    cp "${FS_SOURCE_PATH}/pkginfo.sign" "${ROOT_MOUNT}/usr/share/horizon/pkginfo.sign"
    cp "${FS_SOURCE_PATH}/wired.config" "${ROOT_MOUNT}/etc/connman/wired.config"
    cd -
}

do_happ_perms() {
    chroot $ROOT_MOUNT chown -R happ:happ /data/happ/apps /data/happ/data /data/happ/repo
}

# Ensure all written to disk for reboot
do_restore_backup() {
    if [ -x "${backup_file}" ]; then
      cd "${DATA_MOUNT}"
      tar --use-compress-program=lzop -pxf "${backup_file}"
      cd -
    fi
    sync
}

do_install_brand() {
    cd "${DATA_MOUNT}"
    if [ -x "${FS_SOURCE_PATH}/Console.ini" ]; then
      sed -i "/\[Console\]/d" x30/Console.ini
      sed -i "/Brand=/d" x30/Console.ini
      cat "${FS_SOURCE_PATH}/Console.ini" >> x30/Console.ini
    fi
    cd -

    # Install hostname from hostnames file if provided
    MAC=$( ifconfig eth0 2>/dev/null | grep HWaddr | sed 's/.*HWaddr\ *\([a-zA-Z0-9:]*\).*/\1/' 2>/dev/null)
    if [ ! -z "${MAC}" ]; then
      USB_HOSTNAME=$( grep -i $MAC ${FS_SOURCE_PATH}/hostnames 2>/dev/null | cut -d' ' -f2)
    fi

    if [ ! -z "${USB_HOSTNAME}" ]; then
      # Found MAC address in hostnames file
      echo "${USB_HOSTNAME}" > "${ROOT_MOUNT}/etc/hostname"
      rm "${hostname_file}"
    elif [ -f "${hostname_file}" ]; then 
      # Did not find MAC address in hostnames file update file but have backup hostname file
      if [ -f "${FS_SOURCE_PATH}/hostnames" ]; then
        echo "${MAC} $(cat ${hostname_file})" >> ${FS_SOURCE_PATH}/hostnames
      fi
      mv "${hostname_file}" "${ROOT_MOUNT}/etc/hostname"
    fi

    # if user hostname file is provided, use that as default regardless of what is in hostnames	file
    if [ -f "${FS_SOURCE_PATH}/hostname" ]; then
      cp "${FS_SOURCE_PATH}/hostname" "${ROOT_MOUNT}/etc/hostname"
    fi

    # Install angstrom_feeds if provided
    if [ -f "${FS_SOURCE_PATH}/angstrom_feeds.conf" ]; then
      cp "${FS_SOURCE_PATH}/angstrom_feeds.conf" "${ROOT_MOUNT}/etc/opkg/angstrom_feeds.conf"
    fi

    # Install wired.config if provided
    if [ -f "${FS_SOURCE_PATH}/wired.config" ]; then
      cp "${FS_SOURCE_PATH}/wired.config" "${ROOT_MOUNT}/var/lib/connman/wired.config"
    fi

    do_restore_logo
}

do_install_extras() {
    mkdir /tmp/extras
    cd /tmp/extras

    # Get Machine type from rootfs filename
    MACHINE="$(basename "${FS_SOURCE}" | awk -F- '{
            if ($1 == "egspro" || $1 == "egscore") printf "%s-%s", $1, $2
            else print $1
    }')"

    # Untar extras into tmp folder and install
    tar -xf ${FS_SOURCE_PATH}/${MACHINE}-extras.img
    opkg install -o ${ROOT_MOUNT} $(cat extras.*.txt | sort | uniq)
    chroot ${ROOT_MOUNT} opkg configure
    cd -
}

do_install_hpkgs() {
    export HAPPDIR=${DATA_MOUNT}/happ
    ls ${FS_SOURCE_PATH}/hpkgs/*.hpkg 2>/dev/null | while read pkg
    do
        uid=$(${ROOT_MOUNT}/usr/bin/hpkg info -p $pkg | awk '/^uid:/{print $2}')
        if ! ls ${HAPPDIR}/apps/${uid}/* >/dev/null 2>&1
        then
                ${ROOT_MOUNT}/usr/bin/hpkg install $pkg
        fi
    done
}

do_reboot() {

    do_unmount_target
    umount "${FS_SOURCE_PATH}"

    # poweroff for production build
    # poweroff -f && while true; do sleep 1000; done

    # Remove Topcon signature
    usb_drive_mbr="$(echo ${FS_SOURCE_PATH} | sed -n -r 's/\/media(\/sd[a-z]{1})[0-9]+\//\/dev\1/p')"
    echo -e -n "\000\000\000\000\000\000" | dd of="${usb_drive_mbr}" seek=440 bs=1 count=6
    sync

    # reboot for normal upgrade
    reboot -f && while true; do sleep 1000; done
}

do_backup() {
    # Generate unique file name for the backup
    time=$(date +%s)
    export backup_file="${FS_SOURCE_PATH}/backup_${time}.tar.lzo"
	
    # Backup user data
    cd "${ROOT_MOUNT}/data"
    if [ -d x30 ]; then
        # Build the exclude list file
        echo "x30/.FactoryVehicles*"        > /tmp/exclude_list
        echo "x30/.FactoryImplements*"     >> /tmp/exclude_list
        echo "x30/.FactoryProducts*"       >> /tmp/exclude_list
        echo "x30/.FactoryNozzles*"        >> /tmp/exclude_list
        echo "x30/.usbkey-full-on-upgrade" >> /tmp/exclude_list
        echo "x30/FactoryVehicles*"        >> /tmp/exclude_list
        echo "x30/FactoryImplements*"      >> /tmp/exclude_list
        echo "x30/FactoryProducts*"        >> /tmp/exclude_list
        echo "x30/FactoryNozzles*"         >> /tmp/exclude_list
        echo "x30/oem_profile.conf"        >> /tmp/exclude_list
        echo "x30/UserLevelsFactory.ini"   >> /tmp/exclude_list
        echo "x30/cache"                   >> /tmp/exclude_list
        echo "x30/Trolltech.conf"          >> /tmp/exclude_list
        echo "x30/CrashReports"            >> /tmp/exclude_list
        echo "x30/Diagnostics"             >> /tmp/exclude_list
        echo "x30/Sounds"                  >> /tmp/exclude_list
        echo "x30/RemoteView"              >> /tmp/exclude_list
        echo "x30/backup_*.tar.gz"         >> /tmp/exclude_list
        echo "x30/backup_*.tar.lzo"        >> /tmp/exclude_list
        echo "x30/cache"                   >> /tmp/exclude_list
        echo "x30/agi1-fw"                 >> /tmp/exclude_list
        echo "x30/agi2-fw"                 >> /tmp/exclude_list
        echo "x30/agi3-fw"                 >> /tmp/exclude_list
        echo "x30/agi4-fw"                 >> /tmp/exclude_list
        echo "x30/sgr1-fw"                 >> /tmp/exclude_list
        echo "x30/asc10-fw"                >> /tmp/exclude_list
        echo "x30/Software.ini"            >> /tmp/exclude_list
        echo "x30/*.csexe"                 >> /tmp/exclude_list
        echo "x30/*.disabled"              >> /tmp/exclude_list
        echo "x30/*.enabled"               >> /tmp/exclude_list
        echo "x30/smb.inc"                 >> /tmp/exclude_list

        # if a Console.ini file has been provided, used it
        if [ -x "${FS_SOURCE_PATH}/Console.ini" ]; then
          echo "x30/Console.ini" >> /tmp/exclude_list
        fi		

        dirs=""
        dirs="$dirs $(test -d x30 && echo x30)"
        dirs="$dirs $(test -d happ/data && echo happ/data)"

        tar -X /tmp/exclude_list --use-compress-program=lzop -pcf "${backup_file}" ${dirs}
        rm -f /tmp/exclude_list

        if [ "$?" != "0" ]; then
          rm "${backup_file}"
          touch "x30/.usbkey-full-on-upgrade"
          do_reboot
        fi

        sync
    fi
    cd -
}

do_etc_backup() {
    time=$(date +%s)

    # Backup hostname file
    export hostname_file="/tmp/hostname_${time}"

    if [ ! -f "${FS_SOURCE_PATH}/hostname" ]; then
      cp "${ROOT_MOUNT}/etc/hostname" "${hostname_file}"
    else
      cp "${FS_SOURCE_PATH}/hostname" "${hostname_file}"
    fi

    # Backup angstrom_feeds.conf if not provided on USB key
    if [ ! -f "${FS_SOURCE_PATH}/angstrom_feeds.conf" ]; then
      cp "${ROOT_MOUNT}/etc/opkg/angstrom_feeds.conf" "${FS_SOURCE_PATH}/angstrom_feeds.conf"
    fi

    # Backup wired.config if not provided on USB key
    if [ ! -f "${FS_SOURCE_PATH}/wired.config" ]; then
      cp "${ROOT_MOUNT}/var/lib/connman/wired.config" "${FS_SOURCE_PATH}/wired.config"
    fi

    do_backup_logo
}

do_delete_pclient_project() {
    cd "${ROOT_MOUNT}/data"
    if [ -d pclient ]; then
        # don't want to keep PClient project
        rm -rf pclient/projekte/default_prj/
        sync
    fi
    cd -
}

progressbar() {
    /usr/bin/upgrade-splash $1 2>&1 >/dev/null
}

do_full_install() {
    progressbar 0

    if test -n "$(fstype "$DATA_DEV")"; then
        fsck.ext3 -p "$DATA_DEV" || :

        if ! fsck.ext3 -n "$DATA_DEV" || check_datafs_changed; then
            REFORMAT_DATA=1
        fi

        if test "$REFORMAT_DATA" = 1; then
            do_mount_target
            do_etc_backup
            progressbar 10
            do_backup
            do_unmount_target
        else
            do_mount_target
            do_delete_pclient_project
            do_etc_backup
            do_unmount_target
        fi
    else
        # Check if perhaps the file system isn't ext3
        if fsck.ext3 -n "$DATA_DEV"; then
            do_mount_target
            do_etc_backup
            progressbar 10
            do_backup
            do_unmount_target
        fi

        REFORMAT_DATA=1
    fi

    if test -n "$(fstype "$SHADOW_DEV")"; then
        fsck.ext3 -p "$SHADOW_DEV" || :

        if ! fsck.ext3 -n "$SHADOW_DEV"; then
            REFORMAT_SHADOW=1
        fi
    else
        test ! ${SHADOW_DEV} || REFORMAT_SHADOW=1
    fi

    do_console_details
    progressbar 20

    do_create_partition_table
    progressbar 30
	
    do_create_file_systems
    do_mount_target
    progressbar 40

    do_extract_fs_image
    progressbar 50
	
    do_install_brand
    progressbar 60

    do_install_extras
    progressbar 70

    do_install_hpkgs
    progressbar 80

    do_happ_perms
    progressbar 90

    do_restore_backup
    progressbar 100

    do_set_boot_mode

    do_reboot
}

if [ "$3" == "run" ]; then
    do_check_disk_size
    do_full_install
else
    cp -f "$0" /tmp/install
    chmod +x /tmp/install
    mkdir -p "${LOG_PATH}"
    exec /tmp/install "$1" "$2" run 2>&1 | tee "$LOG_CONSOLE"
fi
